#include <ratio>

template<bool b> struct Bool
{
	using type = bool;
    static const type value = b;
    constexpr operator bool() const { return value; }
};

template<typename T> struct Exist;
template<>           struct Exist<Bool<true>>{ using result = Bool<true>; };

template<int n> struct Int
{
	using type = int;
    static const type value = n;
    constexpr operator int() const { return value; }
    auto operator++()    const { return Int<(n+1)>(); }
    auto operator++(int) const { return Int<(n+1)>(); }
    auto operator--()    const { return Int<(n-1)>(); }
    auto operator--(int) const { return Int<(n-1)>(); }
};

template<int n1, int n2> auto operator+(Int<n1>, Int<n2>){ return Int<n1 + n2>(); }
template<int n1, int n2> auto operator-(Int<n1>, Int<n2>){ return Int<n1 - n2>(); }
template<int n1, int n2> auto operator*(Int<n1>, Int<n2>){ return Int<n1 * n2>(); }
template<int n1, int n2> auto operator/(Int<n1>, Int<n2>){ return Int<n1 / n2>(); }
template<int n1, int n2> auto operator==(Int<n1>, Int<n2>){ return Bool<(n1 == n2)>(); }

template<int64_t n, int64_t d> struct Rat{ static const int64_t nom = n; static const int64_t den = d; operator double() const { return (double)n / (double)d; } };

template<int64_t n1, int64_t n2, int64_t d1, int64_t d2> auto operator+( Rat<n1, d1>, Rat<n2, d2> ){ using res = std::ratio_add< std::ratio<n1, d1>, std::ratio<n2, d2> >; return Rat< res::num, res::den >(); }
template<int64_t n1, int64_t n2, int64_t d1, int64_t d2> auto operator-( Rat<n1, d1>, Rat<n2, d2> ){ using res = std::ratio_subtract< std::ratio<n1, d1>, std::ratio<n2, d2> >; return Rat< res::num, res::den >(); }
template<int64_t n1, int64_t n2, int64_t d1, int64_t d2> auto operator*( Rat<n1, d1>, Rat<n2, d2> ){ using res = std::ratio_multiply< std::ratio<n1, d1>, std::ratio<n2, d2> >; return Rat< res::num, res::den >(); }
template<int64_t n1, int64_t n2, int64_t d1, int64_t d2> auto operator/( Rat<n1, d1>, Rat<n2, d2> ){ using res = std::ratio_divide< std::ratio<n1, d1>, std::ratio<n2, d2> >; return Rat< res::num, res::den >(); }

//Product type:
template<typename N, typename T> struct elem       { T val; elem():val(T()){} elem( T const& v ):val(v){} elem(T && v) :val(std::forward<T>(v)){} };
template<typename N, typename T> struct elem<N, T&>{ T val;                   elem( T const& v ):val(v){}                               };

#define el(n, T) elem< decltype(n), T >

template<typename... Ts> struct List
{
};

template<typename Nk, typename Nn, typename T> struct ProductType_;
template<typename Nk>                          struct ProductType_<Nk, Int<0>, List<>>
{
    template<int i> void operator[]( Int<i> idx)     { }
    template<int i> void operator[]( Int<i> idx)const{ }
};

template<typename Nk, typename Nn, typename T0, typename... T>
struct ProductType_<Nk, Nn, List<T0, T...>> : elem< decltype(--Nk()), T0>, ProductType_< decltype(Nk()++), Nn, List<T...>>
{
    using ProductType_< decltype(Nk()++), Nn, List<T...>>::operator[];
    ProductType_(){}
    ProductType_(ProductType_<Nk, Nn, List<T0, T...>>const& p): elem<decltype(--Nk()), T0>(p[--Nk()]), ProductType_< decltype(Nk()++), Nn, List<T...>>( static_cast<ProductType_< decltype(Nk()++), Nn, List<T...>>const&>(p)){}
    ProductType_(ProductType_<Nk, Nn, List<T0, T...>>&&     p): elem<decltype(--Nk()), T0>(p[--Nk()]), ProductType_< decltype(Nk()++), Nn, List<T...>>( static_cast<ProductType_< decltype(Nk()++), Nn, List<T...>>&&    >(p)){}

    auto& operator= (ProductType_ const& cpy){ (static_cast<elem< decltype(--Nk()), T0>*>(this))->val = (static_cast<const elem< decltype(--Nk()), T0>*>(&cpy))->val; return *this; }

    template<typename T0c, typename... Tsc> ProductType_(T0c && t, Tsc &&... ts) : elem< decltype(--Nk()), T0>( std::forward<T0c>(t) ), ProductType_< decltype(Nk()++), Nn, List<T...>>( std::forward<Tsc>(ts)... ) {}

    T0&       operator[]( decltype(--Nk()) idx)     { return (static_cast<      elem<decltype(--Nk()), T0>*>(this))->val; }
    T0 const& operator[]( decltype(--Nk()) idx)const{ return (static_cast<const elem<decltype(--Nk()), T0>*>(this))->val; }
};

template<typename Nn, typename T> struct ProductType_<Nn, Nn, List<T> > : elem< decltype(--Nn()), T>
{
    ProductType_(){}
    ProductType_(ProductType_<Nn, Nn, List<T>>const& cpy):elem< decltype(--Nn()), T>(cpy[--Nn()]){}
    ProductType_(ProductType_<Nn, Nn, List<T>>&&     cpy):elem< decltype(--Nn()), T>(cpy[--Nn()]){}

    auto& operator= ( ProductType_<Nn, Nn, List<T>> const& cpy){ (static_cast<elem< decltype(--Nn()), T>*>(this))->val = (static_cast<const elem< decltype(--Nn()), T>*>(&cpy))->val; return *this; }

    template<typename Tc> ProductType_(Tc && t):elem< decltype(--Nn()), T>(std::forward<Tc>(t)){}
	T&        operator[]( decltype(--Nn()) idx)     { return (static_cast<      elem< decltype(--Nn()), T>*>(this))->val; }
    T const & operator[]( decltype(--Nn()) idx)const{ return (static_cast<const elem< decltype(--Nn()), T>*>(this))->val; }
};

template<typename... T> struct Tuple
{
	using size = Int<sizeof...(T)>;
    ProductType_<Int<1>, Int<sizeof...(T)>, List<T...>> p;

    Tuple(){}
    Tuple(Tuple<T...> const& pin):p(          pin.p) {}
    Tuple(Tuple<T...> &&     pin):p(std::move(pin.p)){}

    template<typename... Tin> Tuple( int dummy, Tin&&... ts):p( std::forward<Tin>(ts)... ){}

    auto& operator= (Tuple<T...> const& cpy){ p = cpy.p; return *this; }

    template<int i> auto &      operator[]( Int<i> idx)     { return p[idx]; }
    template<int i> auto const& operator[]( Int<i> idx)const{ return p[idx]; }
};

template<typename T>     struct IsTuple;
template<typename... Ts> struct IsTuple< Tuple<Ts...> >{ using result = void; };

template<typename T>     struct IsNotTuple{ using result = void; };
template<typename... Ts> struct IsNotTuple< Tuple<Ts...> >;

template<typename T, typename R = typename IsTuple<std::remove_reference_t<T>>::result> auto size( T const& tuple ){ return typename std::remove_reference_t<T>::size(); }

template<typename... Ts>
auto makeTuple( Ts&&... ts){ return Tuple<std::remove_reference_t<Ts>...>( 0, std::forward<Ts>(ts)... ); }

//Expand
template<typename Ni, typename Nn> struct expand_
{
	template<typename F, typename Tup, typename... A>
	decltype(auto) operator()( F&& f, Tup&& t, A&&... a){ return expand_< decltype(++Ni()), Nn>()( std::forward<F>(f), std::forward<Tup>(t), std::forward<A>(a)..., t[--Ni()] ); }
};

template<typename Nn> struct expand_<Nn, Nn>
{
	template<typename F, typename Tup, typename... A>
	decltype(auto) operator()( F&& f, Tup&& t, A&&... a){ return f( std::forward<A>(a)..., t[--Nn()] ); }
};

template<typename F>
decltype(auto) expand(F&& f, Tuple<> t){ return t; }

template<typename F, typename Tup>
decltype(auto) expand(F&& f, Tup&& t){ return expand_<Int<1>, decltype(size(t))>()(std::forward<F>(f), std::forward<Tup>(t)); }

//Map
template<typename F, typename T, typename R = typename IsTuple<std::remove_reference_t<T>>::result>
auto Map( F&& f, T&& t)
{
	return expand( [&](auto... e){ return makeTuple( f( std::forward<decltype(e)>(e))... ); }, std::forward<T>(t) );
}

//FOLD
template<typename F> struct Foldl_inner_lambda
{
    F f;
	Foldl_inner_lambda(F&& ff):f(std::move(ff)){}
	Foldl_inner_lambda(F const& ff):f(ff){}

    template<typename T1>                                          auto operator()(T1&& t1)         { return std::forward<T1>(t1); }
    template<typename T1, typename T2>                             auto operator()(T1&& t1, T2&& t2){ return f(std::forward<T1>(t1), std::forward<T2>(t2)); }
    template<typename T1, typename T2, typename T3, typename... T> auto operator()(T1&& t1, T2&& t2, T3&& t3, T&&... t)
    {
		return this->operator()( f(std::forward<T1>(t1), std::forward<T2>(t2)), std::forward<T3>(t3), std::forward<T>(t)... );
    }
};

template<typename F, typename T>
auto Foldl(F&& f, T&& t){ return expand( Foldl_inner_lambda<std::remove_cv_t<std::remove_reference_t<F>>>(std::forward<F>(f)), std::forward<T>(t) ); }

//CONCATENATE
template<typename     T, typename   Pck> struct Concat;
template<typename     T, typename... Ts> struct Concat<T, Tuple<Ts...>>           { using result = Tuple<T, Ts...>; };
template<typename... Ts, typename     T> struct Concat<Tuple<Ts...>, T>           { using result = Tuple<Ts..., T>; };
template<typename... As, typename... Bs> struct Concat<Tuple<As...>, Tuple<Bs...>>{ using result = Tuple<As..., Bs...>; };

template<typename T1, typename T2>
auto append( T1&& t1, T2&& t2, typename IsTuple<std::decay_t<T1>>::result* rq1 = nullptr, typename IsTuple<std::decay_t<T2>>::result* rq2 = nullptr )
{
    return expand( [&](auto&&... e2)
    {
        return expand( [&](auto&&... e1)
        {
            return makeTuple( std::forward<decltype(e1)>(e1)..., std::forward<decltype(e2)>(e2)... );
        }, std::forward<T1>(t1) );
    }, std::forward<T2>(t2) );
}

//generate nat sequence:
auto genseq( Int<1> ){ return makeTuple( Int<0>() ); }
template<int n> auto genseq(Int<n> i){ auto k = --i; return append( genseq( k ), makeTuple(k) ); }

//Zip
template<typename F, typename T1, typename T2, typename R = typename Exist<decltype(size(std::declval<T1>()) == size(std::declval<T2>()) )>::result>
auto Zip( F&& f, T1&& t1, T2&& t2)
{
	return expand( [&](auto... ns){ return makeTuple( f( t1[ns], t2[ns] )... ); }, genseq(size(t1)) );
}

//Array
template<typename T, int n> struct Array
{
    using size = Int<n>;
    T data[n];
    template<int n> T&       operator[] (Int<n>)     { return data[n]; }
    template<int n> T const& operator[] (Int<n>)const{ return data[n]; }
                    T&       operator[] (int n)      { return data[n]; }
                    T const& operator[] (int n) const{ return data[n]; }
};

template<typename T, int n>
auto size( Array<T, n> const& ){ return Int<n>(); }

template<typename T0, typename... T>
auto makeArray( T0&& t0, T&&... ts ){ return Array<std::remove_reference_t<T0>, sizeof...(T)+1>{ {std::forward<T0>(t0), std::forward<T>(ts)...} }; }

//Array Slice
template<typename T, int n, int n0, int n1> struct ArraySlice
{
    using size = Int<n1-n0>;
    T (&data)[n];
    template<int i> T&       operator[] (Int<i>)     { return data[n0+i]; }
    template<int i> T const& operator[] (Int<i>)const{ return data[n0+i]; }
                    T&       operator[] (int i)      { return data[n0+i]; }
                    T const& operator[] (int i) const{ return data[n0+i]; }
};

template<typename T, int n, int n0, int n1>
auto size( ArraySlice<T, n, n0, n1> const& ){ return Int<n1-n0>(); }

template<int n0, int n1, typename T, int n>
auto slice( Int<n0> N0, Int<n1> N1, Array<T, n> const& A )
{
    return ArraySlice<const T, n, n0, n1>{ A.data };
}

template<typename... Ts>
auto flatten( Tuple<Ts...> const& tup )
{
    return Foldl( [](auto&& l, auto&& r){ return append(std::forward<decltype(l)>(l), std::forward<decltype(r)>(r)); }, tup );
}

template<typename T, typename... Ts>
auto flattenToArray( Tuple<Ts...> const& tup )
{
    return expand([](auto&&... elems){ return makeArray( ( (T)std::forward<decltype(elems)>(elems))...); }, flatten(tup));
}

//Finite Difference
template<typename Order, typename Coeffs> struct FDScheme
{
    Coeffs coeffs;
};

template<typename O, typename... Cs, typename T, int n>
auto applyFD( FDScheme<O, Tuple<Cs...>> fds, Array<T, n> const& a )
{
    auto cs = Int<sizeof...(Cs)>();
    auto ns = Int<n>();
    auto mul = [](auto x, auto y){ return x*y; };
    auto add = [](auto x, auto y){ return x+y; };
    auto one = Int<1>();
    auto two = Int<2>();

    auto fs = makeTuple(
                      Map( [&](auto c)
                           {
                               return Foldl(add, Zip(mul, fds.coeffs[c], slice(Int<0>(), O(), a) ));
                           }, genseq(O()/Int<2>()) ),
                      Map( [&](auto c)
                           {
                               return Foldl(add, Zip(mul, fds.coeffs[O()/Int<2>()], slice(c, c+O(), a) ));
                           }, genseq(ns-O()+one) ),
                      Map( [&](auto c)
                           {
                               return Foldl(add, Zip(mul, fds.coeffs[cs-(O()-two)+c], slice(ns-O(), ns, a) ));
                           }, genseq(O()/Int<2>()) )
                    );
    return fs;
}

#include <iostream>

template<typename... Ts>
std::ostream& operator<<( std::ostream& s, Tuple<Ts...> const& tup)
{
    struct W{ std::ostream* s; } w{&s};
    s << "{";
    Foldl( [&]( auto&& stream, auto elem ){ (*stream.s) << elem << ", "; return stream; }, append( makeTuple(w), tup ) );
    s << "}";
    return s;
}

template<typename T, int n>
std::ostream& operator<<( std::ostream& s, Array<T, n> const& a)
{
    s << "{";
    s << Foldl( [&]( auto elem1, auto elem2 ){ s << elem1 << ", "; return elem2; }, a ) << "}";
    return s;
}

void main21()
{
    auto A = makeArray( 1, 2, 3, 4, 5, 6, 7, 8 );

    FDScheme<Int<3>, Tuple< Tuple<Int<100>, Int<10>, Int<1>>,
                            Tuple<Int<100>, Int<10>, Int<1>>,
                            Tuple<Int<100>, Int<10>, Int<1>> >> fds1;

    FDScheme<Int<5>, Tuple< Tuple<Int<10000>, Int<1000>, Int<100>, Int<10>, Int<1>>,
                            Tuple<Int<10000>, Int<1000>, Int<100>, Int<10>, Int<1>>,
                            Tuple<Int<10000>, Int<1000>, Int<100>, Int<10>, Int<1>>,
                            Tuple<Int<10000>, Int<1000>, Int<100>, Int<10>, Int<1>>,
                            Tuple<Int<10000>, Int<1000>, Int<100>, Int<10>, Int<1>>  >> fds2;

    auto res = flattenToArray<int>(applyFD(fds1, A));
    std::cout << res << std::endl;

    auto res2 = flattenToArray<int>(applyFD(fds2, A));
    std::cout << res2 << std::endl;
}
